#include <Arduino.h>
#include <time.h>
#include "../core/player.h"
#include "../core/config.h"
#include "../../myoptions.h"
#include "clock_tts.h"
#include "../core/network.h" 

static int clock_tts_prev_volume = 0;
static bool clock_tts_fading_down = false;
static bool clock_tts_fading_up = false;
static unsigned long clock_tts_fade_timer = 0;
static int clock_tts_fade_volume = -1;
static unsigned long clock_lastTTSMillis = 0;
static bool clock_ttsActive = false;
static int clock_lastMinute = -1;
static bool clock_wasPlayingBeforeTTS = false;  // Nowa zmienna: czy grało przed TTS

// Zmienne konfiguracyjne
static bool clock_tts_enabled = CLOCK_TTS_ENABLED;
static int clock_tts_interval = CLOCK_TTS_INTERVAL_MINUTES;
static char clock_tts_language[32] = CLOCK_TTS_LANGUAGE;

void clock_tts_set_language(const char* lang) {
    if (lang && strlen(lang) < sizeof(clock_tts_language)) {
        strcpy(clock_tts_language, lang);
    }
}

void clock_tts_set_interval(int minutes) {
    if (minutes > 0) clock_tts_interval = minutes;
}

void clock_tts_enable(bool enable) {
    clock_tts_enabled = enable;
}

void clock_tts_setup() {
    clock_tts_prev_volume = 0;
    clock_tts_fading_down = false;
    clock_tts_fading_up = false;
    clock_tts_fade_timer = 0;
    clock_tts_fade_volume = -1;
    clock_lastTTSMillis = 0;
    clock_ttsActive = false;
    clock_lastMinute = -1;
    clock_tts_enabled = config.store.clock_tts_enabled; // <- wł/wył TTS (odczyt z konfiguracji)
}

void clock_tts_force(const char* text, const char* lang) {
    player.connecttospeech(text, lang ? lang : clock_tts_language);
    clock_lastTTSMillis = millis();
    clock_ttsActive = true;
}


static void clock_tts_announcement(char* buf, size_t buflen, int hour, int min, const char* lang) {
    if (strncmp(lang, "en", 2) == 0) {
        snprintf(buf, buflen, "The time is %d:%02d.", hour, min);
    } else if (strncmp(lang, "de", 2) == 0) {
        snprintf(buf, buflen, "Es ist %d:%02d Uhr!.", hour, min);
    } else if (strncmp(lang, "ru", 2) == 0) {
        snprintf(buf, buflen, "Сейчас %d:%02d.", hour, min);
    } else if (strncmp(lang, "ro", 2) == 0) {
        snprintf(buf, buflen, "Este ora %d:%02d.", hour, min);
    } else if (strncmp(lang, "hu", 2) == 0) {
        snprintf(buf, buflen, "Az idő %d:%02d.", hour, min);
    } else {
        snprintf(buf, buflen, "Jest godzina %d:%02d.", hour, min);
    }
}

void clock_tts_loop() {
    if (!clock_tts_enabled) return;

    // Blokuj zapowiedź godziny, jeśli odtwarzanie z karty SD
    if (config.getMode() == PM_SDCARD) return;

    unsigned long nowMillis = millis();
    time_t now = time(nullptr);
    struct tm *tm_struct = localtime(&now);

    // --- Stopniowe wyciszanie przed TTS ---
    if (clock_tts_fading_down) {
        if (clock_tts_fade_volume == -1) clock_tts_fade_volume = player.getVolume();
        // 5000 ms / 100 kroków = 50 ms na krok (przyjmujemy 100 poziomów głośności)
        if (nowMillis - clock_tts_fade_timer > 50 && clock_tts_fade_volume > 0) {
            clock_tts_fade_volume -= 1;
            if (clock_tts_fade_volume < 0) clock_tts_fade_volume = 0;
            player.setVolume(clock_tts_fade_volume);
            clock_tts_fade_timer = nowMillis;
        }
        if (clock_tts_fade_volume <= 0) {
            delay(150);
            player.lockOutput = false;
            // ZAPAMIĘTAJ poprzednią głośność
            int prevVol = clock_tts_prev_volume;
            // USTAW głośność na czas komunikatu!
            player.setVolume(prevVol); //aktualna głośność streamu radia z przed komunikatu
            //player.setVolume(50); //głośność ustawiona na stały poziomie
            char buf[48];
            clock_tts_announcement(buf, sizeof(buf), tm_struct->tm_hour, tm_struct->tm_min, clock_tts_language);
            player.connecttospeech(buf, clock_tts_language);
            // PRZYWRÓĆ poprzednią głośność
            player.setVolume(prevVol);
            clock_lastTTSMillis = nowMillis;
            clock_tts_fading_down = false;
            clock_ttsActive = true;
            clock_lastMinute = tm_struct->tm_min;
            clock_tts_fade_volume = -1;
        }
        return;
    }

    // --- Stopniowe pogłaśnianie po TTS ---
    if (clock_tts_fading_up) {
        if (clock_tts_fade_volume == -1) clock_tts_fade_volume = 0; // start od 0
        // 8000 ms / 100 kroków = 80 ms na krok (bardzo płynnie)
        if (nowMillis - clock_tts_fade_timer > 80 && clock_tts_fade_volume < clock_tts_prev_volume) {
            clock_tts_fade_volume += 1;
            player.setVolume(clock_tts_fade_volume);
            clock_tts_fade_timer = nowMillis;
        }
        if (clock_tts_fade_volume >= clock_tts_prev_volume) {
            player.setVolume(clock_tts_prev_volume);
            clock_tts_fading_up = false;
            clock_tts_fade_volume = -1;
        }
        return;
    }

    if (tm_struct) {
        if (tm_struct->tm_year + 1900 < 2020) {
            // Czas niezsynchronizowany, nie ogłaszaj TTS
            return;
        }
        if (
            tm_struct->tm_min % clock_tts_interval == 0 &&
            tm_struct->tm_min != clock_lastMinute &&
            tm_struct->tm_sec < 2 && // Dodano: tylko na początku minuty!
            !clock_ttsActive
        ) {
            clock_wasPlayingBeforeTTS = player.isRunning();  // Zapamiętaj stan
            clock_tts_prev_volume = player.getVolume();
            clock_tts_fading_down = true;
            clock_tts_fade_timer = nowMillis;
            return;
        }
        if (clock_ttsActive && (nowMillis - clock_lastTTSMillis > 4500)) {   // 4500 czas w milisekundach do wznowienia streamu
            if (clock_wasPlayingBeforeTTS) {  // Przywróć radio TYLKO jeśli grało przed TTS
                player.sendCommand({PR_PLAY, config.lastStation()});
                clock_tts_fading_up = true;
                clock_tts_fade_timer = nowMillis;
            }
            clock_ttsActive = false;
            return;
        }
    } else {
        static unsigned long lastTTS = 0;
        if (nowMillis - lastTTS > 120000) {
            player.connecttospeech("Nie można pobrać czasu", clock_tts_language);
            lastTTS = nowMillis;
        }
    }
}